<?php

namespace Modules\Users\Http\Controllers;

use Caffeinated\Flash\Facades\Flash;
use Illuminate\Http\Request;
use Kris\LaravelFormBuilder\FormBuilder;
use Kris\LaravelFormBuilder\FormBuilderTrait;
use Modules\Core\Http\Controllers\Controller;
use Modules\Users\Entities\User;
use Modules\Users\Http\Forms\UsersForm;
use Modules\Users\Services\UserService;

/**
 * Class UsersController
 * @package App\Http\Controllers\Admin
 */
class UsersController extends Controller
{
    use FormBuilderTrait;

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $users = User::where('id', '<>', '1')->get();

        return view('users::users.index', compact('users'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(UsersForm::class, [
            'method' => 'POST',
            'url' => route('users.store'),
        ], ['is_admin' => $this->is_admin()]);

        return view('users::users.create', compact('form'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $form = $this->form(UsersForm::class, [], ['is_admin' => $this->is_admin()]
        );
        $form->validate(['password' => 'required|confirmed|min:3']);
        $form->redirectIfNotValid();

        $user = User::create($form->getFieldValues());
        $user->modules()->sync($request['modules']);
        $user->groups()->sync($request['groups']);
        $user->notification_types()->sync($request['notification_types']);

        return redirect()->route('users.index');
    }

    /**
     * Display the specified resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        return redirect()->route('users.edit', $id);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  User $user
     * @return \Illuminate\Http\Response
     */
    public function edit(User $user, FormBuilder $formBuilder)
    {
        $form = $formBuilder->create(UsersForm::class,
            [
                'method' => 'PATCH',
                'url' => route('users.update', $user),
                'model' => $user->load(['modules', 'groups', 'notification_types'])->toArray()
            ], ['is_admin' => $this->is_admin()]);

        return view('users::users.edit', compact('form'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(User $user, Request $request)
    {
        $form = $this->form(UsersForm::class, [], ['is_admin' => $this->is_admin()]
        );

        $form->redirectIfNotValid();

        $request = $form->getFieldValues();

        if(!$request['password']){
            unset($request['password']);
        }

        $user->update($request);

        $user->modules()->sync($request['modules']);
        $user->groups()->sync($request['groups']);
        $user->notification_types()->sync($request['notification_types']);

        return redirect()->route('users.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, User $user, UserService $service)
    {
        try{
            $service->deleteUser($user);
        }catch (\LogicException $e){
            Flash::error($e->getMessage());
        }

        return redirect()->route('users.index');
    }

    private function is_admin()
    {
        return auth()->user()->is_admin();
    }
}
