<?php

namespace Modules\Core\Http\Controllers;


use Caffeinated\Flash\Facades\Flash;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Modules\Core\Exceptions\ClassNotSpecifiedException;
use Modules\Core\Services\CrudService\CrudServiceFactory;

abstract class CrudController extends Controller
{

    protected $crudService;
    protected $sortOrder = 'DESC';
    protected $sortValue = 'id';
    protected $isSortable = false;
    protected $luminance = false;
    protected $action_url_params = [];
    protected $page_size = 25;
    protected $tabs = [];
    protected $relations = [];
    /**
     * 0 - название списка
     * 1 - название доавления/редактирования
     * @var array
     */
    protected $titles = [];

    public function __construct()
    {
        $this->crudService = CrudServiceFactory::makeCrudService(
            $this->getModel(),
            $this->getForm(),
            $this->tabs,
            $this->relations,
            $this->isSortable,
            $this->luminance
        );
    }

    /**
     * Templates
     */
    protected $templateIndex = 'core::common.index';
    protected $templateAdd = 'core::common.create';
    protected $templateEdit = 'core::common.edit';


    protected function getModel()
    {
        return $this->getClassName('Entities', '', 'Не указан класс модели.');

    }

    protected function getForm()
    {
        return $this->getClassName('Http\\Forms', 'Form', 'Не указан класс формы.');
    }

    protected function getTitle($section = 'index')
    {
        $titles = [
            'index' => $this->titles[0] ?? 'Список',
            'create' => 'Добавление '.($this->titles[1] ?? ''),
            'edit' => 'Редактирование '.($this->titles[1] ?? '')
        ];

        return $titles[$section];
    }


    protected function listFields()
    {
        return [
            'title' => [
                'title' => 'Название',
                'link' =>''
            ],
        ];
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $items = $this->crudService->getItems($this->page_size, $this->sortValue, $this->sortOrder);

        return view($this->templateIndex, [
            'items' => $items,
            'title' => $this->getTitle(),
            'controller' => $this->getController(),
            'fields' => $this->listFields(),
            'sortable' => $this->isSortable
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $route = action($this->getActionRoute('store'), $this->action_url_params);

        $form = $this->crudService->createForm($route);

        return view($this->templateAdd, [
            'form' => $form,
            'title' => $this->getTitle('create'),
            'module_title' => $this->getTitle('index'),
            'controller' => $this->getController(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $this->crudService->store();

        Flash::success('Запись добавлена.');

        return $this->redirectToAction('index', $this->action_url_params);
    }

    /**
     * Display the specified resource.
     *
     * @param $id
     *
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        return $this->redirectToAction('edit', array_merge(['id' => $id], $this->action_url_params));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $item = $this->crudService->getItemById($id);
        $route = action($this->getActionRoute('update'), array_merge(['id' => $id], $this->action_url_params));
        $form = $this->crudService->getEditForm($item, $route);

        return view($this->templateEdit, [
            'form' => $form,
            'title' => $this->getTitle('edit'),
            'module_title' => $this->getTitle('index'),
            'controller' => $this->getController(),
            'item' => $item,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param $id
     *
     * @return \Illuminate\Http\Response
     */
    public function update($id, Request $request)
    {
        $this->crudService->update($id, $request);

        Flash::success('Запись обновлена.');

        return $this->redirectToAction('index', $this->action_url_params);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $this->crudService->destroy($id);

        Flash::success('Запись удалена.');

        return $this->redirectToAction('index', $this->action_url_params);
    }

    protected function getController()
    {
        return '\\'.get_class($this);
    }

    protected function getActionRoute($action)
    {
        return $this->getController().'@'.$action;
    }

    protected function redirectToAction($action, $parameters = [], $status = '302')
    {
        return redirect()->action($this->getActionRoute($action), $parameters, $status);
    }

    private function getClassName($path, $suffix = '', $error_msg = 'Не указан класс')
    {
        $controller = new \ReflectionClass($this);
        $name = Str::singular(str_replace('Controller', '', $controller->getShortName()));
        $namespace = str_replace('Http\\Controllers', '', $controller->getNamespaceName());

        $class = $namespace.$path.'\\'.$name.$suffix;

        if(!class_exists($class)){
            throw new \Exception($error_msg);
        }

        return $class;
    }
}
