<?php

namespace Modules\Core\Services\CrudService;


use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Http\Request;
use Kris\LaravelFormBuilder\FormBuilder;

class CrudService
{
    use LuminanceTrait;
    use RelationTrait;

    private $model;
    private $sortable;
    private $luminance;
    private $formBuilder;
    private $form;
    private $tabs;
    private $relations;

    public function __construct($model, $form_name, $sortable = false, $relations = null, $luminance = false, $tabs, FormBuilder $formBuilder)
    {
        $this->model = $model instanceof Model || $model instanceof Relation ? $model : new $model;
        $this->sortable = $sortable;
        $this->luminance = $luminance;
        $this->formBuilder = $formBuilder;
        $this->form = $form_name;
        $this->tabs = $tabs;
        $this->relations = $relations;
    }

    public function getItems($amount, $sort_value = 'id', $sort_order = 'DESC')
    {
        if($this->sortable){
            $model = $this->model->orderBy('pos', 'ASC');
        }else{
            $model = $this->model->orderBy($sort_value, $sort_order);
        }

        return $model->paginate($amount);
    }

    public function createForm($route, $model = [])
    {
        return $this->formBuilder->create($this->form, [
            'method' => 'POST',
            'url' => $route,
            'model' => $model
        ]);
    }

    public function store()
    {
        $values = $this->getFormValues();

        foreach($this->tabs as $tab) {
            if(isset($values[$tab])) {
                $values += $values[$tab];
                unset($values[$tab]);
            }
        }

        if($this->sortable){
            $values['pos'] = $this->model->max('pos')+1;
        }
        if(isset($values['slug']) && empty($values['slug'])) {
            $values['slug'] = \Str::slug($values['title']);
        }

        $item = $this->model->create($this->modelValues($values));

        $this->storeModelRelations($item, $this->relationValues($values));

        return $item;
    }

    private function storeModelRelations(Model $item, $values)
    {
        foreach ($values as $name=>$value) {
            $item->$name()->attach($value);
        }
    }

    public function getItemById($id)
    {
        $model = $this->model;

        if($this->relations){
            $model = $this->model->with($this->relations);
        }

        return $model->findOrFail($id);
    }

    public function getEditForm(Model $item, $route)
    {
        $values = $item->toArray();

        if(property_exists($item, 'translatedAttributes')){
            $values = array_merge($values, $item->getTranslationsArray());
        }

        return $this->formBuilder->create($this->form,
            [
                'method' => 'PATCH',
                'url' => $route,
                'model' => $values,
            ]);
    }

    public function update($id, Request $request) : Model
    {
        $values = $this->getFormValues();

        if($this->luminance && $request->hasFile('image')){
            if($values['image']) {
                $values['luminance'] = $this->getImageBlackOrWhite(storage_path('app/'.$values['image']));
            } else {
                $values['luminance'] = 1;
            }
        }

        $item = $this->model->find($id);
        $item->update($values);
        $this->updateRelations($item, $this->relationValues($values));

        return $item;
    }

    private function updateRelations(Model $item, $values)
    {
        foreach ($values as $name => $value) {
            $item->$name()->sync($value);
        }
    }


    public function destroy($id)
    {
        return $this->model->find($id)->delete();
    }

    public function getFormValues()
    {
        $form = $this->formBuilder->create($this->form);

        $form->redirectIfNotValid();
        $values = $form->getFieldValues();

        return $values;
    }
}